program NEWTONPOLYNOMIAL;
{--------------------------------------------------------------------}
{  Alg4'5.pas   Pascal program for implementing Algorithm 4.5        }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 4.5  (Newton Interpolation Polynomial).                 }
{  Section   4.4,  Newton Polynomials, Page 234                      }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    GNmax = 630;
    MaxN = 50;
    FunMax = 6;

  type
    MATRIX = array[0..MaxN, 0..MaxN] of real;
    VECTOR = array[0..MaxN] of real;
    RVECTOR = array[0..GNmax] of real;
    LETTER = string[8];
    LETTERS = string[200];
    STATUS = (Computing, Done, More, Working);
    DATYPE = (DatPoints, FunPoints);
    ABTYPE = (Given, Equal, Interval, Chebyshev);

  var
    FunType, GNpts, Inum, M, N, Sub: integer;
    A0, B0, Rnum, T: real;
    D: MATRIX;
    A, C, X, Y: VECTOR;
    Xg, Yg: RVECTOR;
    Ans, Resp: CHAR;
    Mess: LETTERS;
    Stat, State: STATUS;
    Ytype: DATYPE;
    Xtype: ABTYPE;

  function F (X: real): real;
  begin
    case FunType of
      1: 
        F := EXP(X);
      2: 
        F := COS(X);
      3: 
        F := SIN(X);
      4: 
        F := SIN(X) / COS(X);
      5: 
        F := ARCTAN(X);
      6: 
        F := LN(X + 1);
    end;
  end;

  procedure PRINTFUN (FunType: integer);
  begin
    case FunType of
      1: 
        WRITE('EXP(X)');
      2: 
        WRITE('COS(X)');
      3: 
        WRITE('SIN(X)');
      4: 
        WRITE('TAN(X)');
      5: 
        WRITE('ARCTAN(X)');
      6: 
        WRITE('LN(X+1)');
    end;
  end;

  procedure DIVIDEDDIFF (X, Y: VECTOR; var D: MATRIX; N: integer);
    var
      J, K: integer;
  begin
    for K := 0 to N do
      D[K, 0] := Y[K];
    for J := 1 to N do
      begin
        for K := J to N do
          D[K, J] := (D[K, J - 1] - D[K - 1, J - 1]) / (X[K] - X[K - J]);
      end;
  end;

  function P (D: MATRIX; X: VECTOR; N: integer; T: real): real;
    var
      K: integer;
      Sum: real;
  begin
    Sum := D[N, N];
    for K := N - 1 downto 0 do
      Sum := Sum * (T - X[K]) + D[K, K];
    P := Sum;
  end;

  procedure MAKEXPOLY (D: MATRIX; X: VECTOR; var C: VECTOR; N: integer);
    var
      J, K: integer;
      Z: real;
  begin
    C[0] := D[N, N];
    for K := 1 to N do
      begin
        Z := X[N - K];
        C[K] := C[K - 1];
        for J := K - 1 downto 1 do
          C[J] := C[J - 1] - Z * C[J];
        C[0] := -Z * C[0] + D[N - K, N - K];
      end;
  end;

  procedure GETFUNCTION (var FunType: integer);
    var
      K: integer;
  begin
    FunType := 0;
    while FunType = 0 do
      begin
        CLRSCR;
        WRITELN;
        WRITELN;
        WRITELN('     The Newton polynomial will be constructed using the function:');
        WRITELN;
        WRITELN;
        for K := 1 to FunMax do
          begin
            WRITE('     <', K : 2, ' >  F(X) = ');
            PRINTFUN(K);
            WRITELN;
            WRITELN;
          end;
        WRITELN;
        WRITELN;
        Mess := '            SELECT < 1 - 6 > ?  ';
        FunType := 1;
        WRITE(Mess);
        READLN(FunType);
        if FunType < 1 then
          FunType := 1;
        if FunType > 6 then
          FunType := 6;
      end;
  end;

  procedure PRINTPOLY (D: MATRIX; X, Y: VECTOR; N: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN;
    if Ytype = FunPoints then
      begin
        WRITE('F(X) = ');
        PRINTFUN(FunType);
        WRITELN;
        WRITELN;
      end;
    WRITELN('The Newton polynomial of degree ', N : 2, ' is:');
    WRITELN;
    case N of
      0:
        begin
          WRITELN('P(x) = a ');
          WRITELN('        0');
        end;
      1:
        begin
          WRITELN('P(x) = a  + a [x - x ]');
          WRITELN('        0    1      0 ');
        end;
      2:
        begin
          WRITELN('P(x) = a  + a [x - x ] + a [x - x ][x - x ]');
          WRITELN('        0    1      0     2      0       1 ');
        end;
      3:
        begin
          WRITELN('P(x) = a  + a [x - x ] + a [x - x ][x - x ] + a [x - x ][x - x ][x - x ]');
          WRITELN('        0    1      0     2      0       1     3      0       1       2 ');
        end;
      4, 5, 6, 7, 8, 9:
        begin
          WRITELN('P(x) = a  + a [x - x ] + a [x - x ][x - x ] + a [x - x ][x - x ][x - x ]');
          WRITELN('        0    1      0     2      0       1     3      0       1       2 ');
          WRITELN;
          WRITELN('      +...+ a [x - x ][x - x ]...[x - x ]');
          WRITELN('             ', N : 1, '      0       1          ', N - 1 : 1);
        end;
      10:
        begin
          WRITELN('P(x) = a  + a [x - x ] + a [x - x ][x - x ] + a [x - x ][x - x ][x - x ]');
          WRITELN('        0    1      0     2      0       1     3      0       1       2 ');
          WRITELN;
          WRITELN('      +...+ a  [x - x ][x - x ]...[x - x ]');
          WRITELN('             ', N : 2, '      0       1          ', N - 1 : 1);
        end;
      else
        begin
          WRITELN('P(x) = a  + a [x - x ] + a [x - x ][x - x ] + a [x - x ][x - x ][x - x ]');
          WRITELN('        0    1      0     2      0       1     3      0       1       2 ');
          WRITELN;
          WRITELN('      +...+ a  [x - x ][x - x ]...[x - x  ]');
          WRITELN('             ', N : 2, '      0       1          ', N - 1 : 2);
        end;
    end;
    WRITELN;
    if N < 10 then
      begin
        WRITE('The coefficients:            The abscissas:');
        WRITELN('             The ordinates:');
        WRITELN;
        for K := 0 to N do
          begin
            WRITE('a(', K : 1, ')  =', D[K, K] : 18 : 8, '    x(', K : 1, ')  =', X[K] : 16 : 6);
            WRITELN('    y(', K : 1, ')  =', Y[K] : 16 : 6);
          end;
      end
    else
      begin
        WRITE('The coefficients:            The abscissas:');
        WRITELN('            The ordinates:');
        WRITELN;
        for K := 0 to N do
          begin
            WRITE('a(', K : 2, ')  =', D[K, K] : 18 : 8, '   x(', K : 2, ')  =', X[K] : 15 : 6);
            WRITELN('   y(', K : 2, ')  =', Y[K] : 15 : 6);
          end;
      end;
  end;

  procedure PRINTXPOLY (A: VECTOR; N: integer);
    var
      K, U, V: integer;
  begin
    CLRSCR;
    WRITELN;
    if Ytype = FunPoints then
      begin
        WRITE('F(X) = ');
        PRINTFUN(FunType);
        WRITELN;
        WRITELN;
      end;
    WRITELN('     When written as an ordinary polynomial,');
    WRITELN('     the polynomial approximation of degree  N = ', N : 2, '  is:');
    WRITELN;
    case N of
      1: 
        begin
          WRITELN('P(X)  =  a  +  a X');
          WRITELN('          0     1');
        end;
      2: 
        begin
          WRITELN('                           2');
          WRITELN('P(X)  =  a   +  a X  +  a X');
          WRITELN('          0      1       2');
        end;
      3: 
        begin
          WRITELN('                           2        3');
          WRITELN('P(X)  =  a   +  a X  +  a X  +  a  X');
          WRITELN('          0      1       2       3');
        end;
      4, 5, 6, 7, 8, 9: 
        begin
          WRITELN('                           2            ', N - 1 : 1, '        ', N : 1);
          WRITELN('P(X)  =  a   +  a X  +  a X   +...+  a X   +  a X');
          WRITELN('          0      1       2            ', N - 1 : 1, '        ', N : 1);
        end;
      10: 
        begin
          WRITELN('                           2            ', N - 1 : 1, '         ', N : 2);
          WRITELN('P(X)  =  a   +  a X  +  a X   +...+  a X   +  a  X');
          WRITELN('          0      1       2            ', N - 1 : 1, '        ', N : 2);
        end;
      else
        begin
          WRITELN('                           2             ', N - 1 : 2, '        ', N : 2);
          WRITELN('P(X)  =  a   +  a X  +  a X   +...+  a  X   +  a  X');
          WRITELN('          0      1       2            ', N - 1 : 2, '        ', N : 2);
        end;
    end;
    WRITELN;
    for K := 0 to TRUNC(N / 2) do                {Print the coefficients}
      begin
        U := 2 * K;                                {of P(X) in two columns}
        V := 2 * K + 1;
        if U <= N then
          begin
            WRITE('A(', U : 2, ' ) =', A[U] : 15 : 7, '         ');
            if V <= N then
              WRITELN('A(', V : 2, ' ) =', A[V] : 15 : 7)
            else
              WRITELN;
          end;
      end;
  end;

  procedure GETPOINTS (var X, Y: VECTOR; var N: integer; Stat: STATUS);
    type
      CONDTS = (Bad, Enter, Done);
      LETTER = string[1];
    var
      I, J, K, Kbad: integer;
      T, Valu: real;
      Resp: CHAR;
      Cond: CONDTS;
  begin
    CLRSCR;
    Kbad := -1;
    if Stat = More then
      begin
        for I := 1 to 6 do
          WRITELN;
        WRITE('Do you want to enter new data points ?  <Y/N>  ');
        READLN(Resp);
        WRITELN;
        if (Resp = 'Y') or (Resp = 'y') then
          begin
            Stat := Working;
            CLRSCR;
          end;
      end;
    if (Stat = Working) then
      begin
        for I := 1 to 6 do
          WRITELN;
        if Xtype = Given then
          begin
            WRITELN('               The Lagrange polynomial will have degree  N,');
            WRITELN;
            WRITELN('               and there must be  N+1  points.');
            WRITELN;
            Mess := '               ENTER the degree   N = ';
            N := 1;
            WRITE(Mess);
            READLN(N);
            if N < 1 then
              N := 1;
            if N > 50 then
              N := 50;
          end;
        for K := 0 to N do
          begin
            if Xtype = Given then
              X[K] := 0;
            Y[K] := 0;
          end;
          CLRSCR;
        WRITELN;
        WRITELN;
        WRITELN;
        WRITELN('         Now  ENTER  the ', N + 1, ' points');
        WRITELN;
        WRITELN('     You will have a chance to make changes at the end.');
        WRITELN;
        WRITELN;
        for K := 0 to N do
          begin
            if Xtype = Given then
              begin
                WRITELN;
                Mess := '         x';
                WRITE(Mess, K : 1, ' = ');
                READLN(X[K]);
              end;
            if Ytype = DatPoints then
              begin
                if Xtype <> Given then
                  begin
                    WRITELN;
                    WRITELN('         x  =', X[K] : 15 : 7);
                    WRITE('          ', K : 0);
                  end;
                WRITELN;
                Mess := '         y';
                WRITE(Mess, K : 1, ' = ');
                READLN(Y[K]);
                WRITELN;
              end
            else
              begin
                Y[K] := F(X[K]);
              end;
            WRITELN;
          end;
      end;
    Cond := Enter;
    while (Cond = Enter) or (Cond = Bad) do
      begin
        CLRSCR;
        if (Cond = Bad) then
          WRITELN('     The abscissas are NOT distinct.   You MUST change one of them.');
        WRITELN('      k               x                     y');
        WRITELN('                       k                     k');
        WRITELN('----------------------------------------------------------------');
        for K := 0 to N do
          begin
            if N < 7 then
              WRITELN;
            WRITELN('     ', K : 2, '       ', X[K] : 15 : 7, '       ', Y[K] : 15 : 7);
          end;
        WRITELN;
        if (Cond <> Bad) then
          begin
            WRITELN;
            if N > 15 then
              begin
                WRITELN;
              end;
            WRITE('     Are the points o.k. ?  <Y/N>  ');
            READLN(Resp);
            WRITELN;
          end;
        if (Resp = 'N') or (Resp = 'n') or (Cond = Bad) then
          begin
            if N > 14 then
              begin
                WRITELN;
              end;
            WRITELN;
            case N of
              1: 
                WRITELN('     To change a point select  k = 0,1');
              2: 
                WRITELN('     To change a point select  k = 0,1,2');
              else
                WRITELN('     To change a point select  k = 0,1,...,', N : 2);
            end;
            Mess := '                       ENTER   k = ';
            K := Kbad;
            WRITE(Mess);
            READLN(K);
            if (0 <= K) and (K <= N) then
              begin
                WRITELN;
                if K < 10 then
                  begin
                    WRITELN('     Coordinates of the  current point  (x ,y )  are:');
                    WRITELN('                                          ', K : 1, '  ', k : 1);
                    WRITELN('     Old   x  =', X[K] : 15 : 7, '     Old   y  =', Y[K] : 15 : 7);
                    WRITELN('            ', K : 1, '                             ', K : 1);
                  end
                else
                  begin
                    WRITELN('     Coordinates of the current point  (x  ,y  )  are:');
                    WRITELN('                                         ', K : 2, '  ', k : 2);
                    WRITELN('     Old   x  =', X[K] : 15 : 7, '     Old   y  =', Y[K] : 15 : 7);
                    WRITELN('            ', K : 2, '                            ', K : 2);
                  end;
                Mess := '     NEW   x';
                WRITE(Mess, K : 1, ' = ');
                READLN(X[K]);
                Mess := '     NEW   y';
                WRITE(Mess, K : 1, ' = ');
                READLN(Y[K]);
                WRITELN;
              end;
          end
        else
          Cond := Done;
        for J := 0 to N - 1 do
          begin
            for K := J + 1 to N do
              if X[J] > X[K] then
                begin
                  T := X[J];
                  X[J] := X[K];
                  X[K] := T;
                  T := Y[J];
                  Y[J] := Y[K];
                  Y[K] := T;
                end;
          end;
        if (Cond = Bad) then
          Cond := Enter;
        Kbad := -1;
        for J := 0 to N - 1 do
          for K := J + 1 to N do
            if (X[J] = X[K]) then
              begin
                Kbad := K;
                Cond := Bad;
              end;
      end;
  end;

  procedure MESSAGE (var FunType: integer; var Xtype: ABTYPE; var Ytype: DATYPE; var X: VECTOR; var N: integer);
    var
      I, J: integer;
      A0, B0, C0, D0, H: real;
      Ans: CHAR;
  begin
    CLRSCR;
    WRITELN('                          NEWTON POLYNOMIALS');
    WRITELN;
    WRITELN('               The Newton polynomial is constructed');
    WRITELN;
    WRITELN('P(x) = a  + a [x - x ] + a [x - x ][x - x ] + a [x - x ][x - x ][x - x ]');
    WRITELN('        0    1      0     2      0       1     3      0       1       2 ');
    WRITELN;
    WRITELN('       +...+ a [x - x ][x - x ]...[x - x   ]');
    WRITELN('              N      0       1          N-1 ');
    WRITELN;
    WRITELN('based on the N+1 points (x ,y ) , (x ,y ) ,... ,(x ,y ) .');
    WRITELN('                          0  0      1  1          N  N');
    WRITELN;
    WRITELN('The centers are the abscissas  x  , x  ,..., x    .    The coefficients');
    WRITELN('                                0    1        N-1 ');
    WRITELN;
    WRITELN('a  , a  ,..., a    , a  , are computed by constructing a divided');
    WRITELN(' 0    1        N-1    N ');
    WRITELN;
    WRITELN('difference table.');
    WRITELN;
    WRITELN;
    WRITE('                          Press the <ENTER> key. ');
    READLN(Ans);
    WRITELN;
    CLRSCR;
    WRITELN;
    WRITELN;
    WRITELN('               You can choose how to enter the ordinates {y }.');
    WRITELN('                                                           k');
    WRITELN;
    WRITELN('          <1>  Enter each value  y  as data.');
    WRITELN('                                  k ');
    WRITELN;
    WRITELN('          <2>  Use a function to compute  y  =  F(x ) .');
    WRITELN('                                           k       k');
    WRITELN;
    Mess := '               SELECT <1 - 2>  ';
    I := 1;
    WRITE(Mess);
    READLN(I);
    if I <= 1 then
      Ytype := DatPoints
    else
      Ytype := FunPoints;
    CLRSCR;
    WRITELN;
    WRITELN;
    WRITELN('          You can choose how to enter the abscissas  {x }.');
    WRITELN('                                                       k');
    WRITELN;
    WRITELN('     <1>  Enter each value  x   as data.');
    WRITELN('                             k  ');
    WRITELN;
    WRITELN('     <2>  Construct  x  = x  + h  for  k = 1,2,...,N.');
    WRITELN('                      k    0');
    WRITELN;
    WRITELN('     <3>  Construct equally spaced points over the interval  [A,B],');
    WRITELN;
    WRITELN('          x  = x  + h   for  k = 1,2,...,N      and      h = (B-A)/N.');
    WRITELN('           k    0');
    WRITELN;
    WRITELN('     <4>  The abscissas will be the Chebyshev nodes for the interval [A,B].');
    WRITELN;
    WRITELN;
    WRITELN;
    Mess := '          SELECT <1 - 4>  ';
    I := 1;
    WRITE(Mess);
    READLN(I);
    if I < 1 then
      I := 1;
    if I > 4 then
      I := 4;
    if I = 1 then
      Xtype := Given;
    if I = 2 then
      begin
        Xtype := Equal;
        CLRSCR;
        WRITELN;
        WRITELN;
        WRITELN('You chose to construct  x  = x  + h  for  k = 1,2,...,N.');
        WRITELN('                         k    0');
        Mess := 'ENTER the starting value  x0 = ';
        X[0] := 0;
        WRITE(Mess);
        READLN(X[0]);
        A0 := X[0];
        Mess := 'ENTER   the  step  size   h = ';
        H := 1;
        WRITE(Mess);
        READLN(H);
        Mess := 'ENTER the number of steps N = ';
        N := 1;
        WRITE(Mess);
        READLN(N);
      end;
    if I = 3 then
      begin
        Xtype := Interval;
        CLRSCR;
        WRITELN;
        WRITELN;
        WRITELN('You chose to construct equally spaced points over the interval [A,B],');
        WRITELN;
        WRITELN('     x  = x  + h   for  k = 1,2,...,N      and      h =(B-A)/N.');
        WRITELN('      k    0');
        Mess := 'ENTER the  left  endpoint  A = ';
        A0 := 0;
        WRITE(Mess);
        READLN(A0);
        Mess := 'ENTER the  right endpoint  B = ';
        B0 := 1;
        WRITE(Mess);
        READLN(B0);
        Mess := 'ENTER the number of steps  N = ';
        N := 1;
        WRITE(Mess);
        READLN(N);
        H := (B0 - A0) / N;
      end;
    if (I = 2) or (I = 3) then
      begin
        for J := 0 to N do
          X[J] := A0 + J * H;
      end;
    if I = 4 then
      begin
        Xtype := Chebyshev;
        CLRSCR;
        WRITELN;
        WRITELN;
        WRITELN('You chose to use the Chebyshev nodes over the interval  [A,B],');
        WRITELN;
        WRITELN('     x  = (A+B)/2 + z (A-B)/2  for  k = 1,2,...,N  where');
        WRITELN('      k              k');
        WRITELN;
        WRITELN('                    z  = cos( (2k+1)Pi/(2N+2) ).');
        WRITELN('                     k ');
        Mess := 'ENTER the  left  endpoint  A = ';
        A0 := 0;
        WRITE(Mess);
        READLN(A0);
        Mess := 'ENTER the  right endpoint  B = ';
        B0 := 1;
        WRITE(Mess);
        READLN(B0);
        WRITELN;
        Mess := 'ENTER the number of steps  N = ';
        N := 1;
        WRITE(Mess);
        READLN(N);
        C0 := (A0 + B0) / 2;
        D0 := (A0 - B0) / 2;
        for J := 0 to N do
          begin
            X[J] := C0 + D0 * COS((2 * J + 1) * PI / (2 * N + 2));
          end;
      end;
    if Ytype = FunPoints then
      GETFUNCTION(FunType);
  end;

  procedure EPOINTS (var A0, B0: real; var M: integer);
    type
      STATUS = (Change, Enter, Done);
      LETTER = string[1];
    var
      Valu: real;
      Ans, Resp: CHAR;
      Stat: STATUS;
  begin
    Stat := Enter;
    while (Stat = Enter) or (Stat = Change) do
      begin
        if (Stat = Enter) then
          begin
            CLRSCR;
            WRITELN;
            WRITELN;
            WRITELN('                    The minimum abscissa is  ', A0 : 15 : 7);
            WRITELN;
            WRITELN('                    The maximum abscissa is  ', B0 : 15 : 7);
            WRITELN;
            WRITELN;
            Mess := '                 ENTER the left  endpoint  A = ';
            WRITE(Mess);
            READLN(A0);
            Mess := '                 ENTER the right endpoint  B = ';
            WRITE(Mess);
            READLN(B0);
            Mess := '                 ENTER number of intervals M = ';
            M := 5;
            WRITE(Mess);
            READLN(M);
            if M < 1 then
              M := 1;
            if M > 100 then
              M := 100;
          end
        else
          begin
            CLRSCR;
            WRITELN;
            WRITELN('                    The left  endpoint is  A =', A0 : 15 : 7);
            WRITELN;
            WRITELN('                    The right endpoint is  B =', B0 : 15 : 7);
            WRITELN;
            WRITELN('                    The number intervals   M = ', M : 2);
            WRITELN;
          end;
        WRITELN;
        WRITE('        Do you want to make a change ?  <Y/N>  ');
        READLN(Ans);
        WRITELN;
        if (Resp = 'Y') or (Resp = 'y') then
          begin
            Stat := Change;
            WRITELN;
            WRITELN('     The current left  endpoint is A =', A0 : 15 : 7);
            Mess := '     ENTER  the NEW left  endpoint A = ';
            WRITE(Mess);
            READLN(A0);
            WRITELN;
            WRITELN('     The current right endpoint is B =', B0 : 15 : 7);
            Mess := '     ENTER  the NEW right endpoint B = ';
            WRITE(Mess);
            READLN(B0);
            WRITELN;
            WRITELN('        The current value of M is  M = ', M : 2);
            Mess := '        ENTER  the  NEW  value of  M = ';
            WRITE(Mess);
            READLN(N);
            if (M < 1) then
              M := 1;
            if M > 100 then
              M := 100;
          end
        else
          Stat := Done;
      end;
  end;

  procedure EVALUATE (D: MATRIX; X: VECTOR; N: integer; var T, A0, B0: real; var M: integer);

    var
      Echoice, J: integer;
      H, Valu: real;
  begin
    A0 := X[0];
    B0 := X[0];
    for J := 1 to N do
      begin
        if A0 > X[J] then
          A0 := X[J];
        if B0 < X[J] then
          B0 := X[J];
      end;
    CLRSCR;
    WRITELN;
    WRITELN('     Now evaluate the Newton polynomial P(x).');
    WRITELN;
    WRITELN(' <1> Enter a single value  x.');
    WRITELN;
    WRITELN(' <2> Compute a table of values over [A,B].');
    WRITELN;
    WRITELN(' <3> Do not evaluate  P(x).');
    WRITELN;
    Mess := '     SELECT <1 - 2>  ';
    Echoice := 1;
    WRITE(Mess);
    READLN(Echoice);
    if (Echoice <> 1) and (Echoice <> 2) then
      Echoice := 3;
    if Echoice = 1 then
      begin
        CLRSCR;
        WRITELN;
        WRITELN('Now evaluate  P(x).');
        Mess := 'ENTER a value   x = ';
        T := 0;
        WRITE(Mess);
        READLN(T);
      end;
    if Echoice = 2 then
      EPOINTS(A0, B0, M);
    if Echoice = 1 then
      begin
        PRINTPOLY(D, X, Y, N);
        WRITELN;
        WRITELN('The value of the Newton polynomial is:');
        WRITELN;
        WRITELN(' P(', T : 15 : 7, '  )  = ', P(D, X, N, T) : 15 : 7);
      end;
    if Echoice = 2 then
      begin
        H := (B0 - A0) / M;
        CLRSCR;
        WRITELN('The values for the Newton polynomial are:');
        WRITELN;
        WRITELN('         x                   P(x)');
        WRITELN('----------------------------------------');
        WRITELN;
        for J := 0 to M do
          begin
            T := A0 + H * J;
            if J = M then
              T := B0;
            WRITELN(T : 15 : 7, '     ', P(D, X, N, T) : 15 : 7);
            if M < 9 then
              WRITELN;
          end
      end;
    if Echoice = 3 then
      WRITELN;
  end;

  procedure DDTABLE (D: MATRIX; X: VECTOR; N: integer);
    var
      J, JB, K, KR: integer;
      H, Valu: real;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('     The divided difference table is:');
    WRITELN;
    WRITELN('       x                 F[x ]             F[ , ]            F[ , , ]          F[ , , , ]');
    WRITELN('        k                   k');
    WRITELN;
    if N <= 7 then
      JB := N
    else
      JB := 7;
    for J := 0 to JB do
      begin
        if J <= 3 then
          KR := J
        else
          KR := 3;
        WRITE(X[J] : 15 : 7, '   ');
        for K := 0 to KR - 1 do
          WRITE(D[J, K] : 15 : 7, '   ');
        if KR <= 3 then
          WRITE(D[J, KR] : 15 : 7);
        WRITELN;
      end;
    if 4 <= N then
      begin
        WRITELN;
        WRITELN('       x                 F[ , , , , ]      F[,,,,,]          F[,,,,,,]         F[,,,,,,,]');
        WRITELN('        k');
        WRITELN;
        for J := 4 to JB do
          begin
            if J <= 7 then
              KR := J
            else
              KR := 7;
            WRITE(X[J] : 15 : 7, '   ');
            for K := 4 to KR - 1 do
              WRITE(D[J, K] : 15 : 7, '   ');
            if KR <= 7 then
              WRITE(D[J, KR] : 15 : 7);
            WRITELN;
          end;
      end;
  end;                                          {End procedure DDTABLE}

begin                                            {Begin Main Program}
  Stat := Working;
  MESSAGE(FunType, Xtype, Ytype, X, N);
  while (Stat = Working) or (Stat = More) do
    begin
      GETPOINTS(X, Y, N, Stat);
      DIVIDEDDIFF(X, Y, D, N);
      PRINTPOLY(D, X, Y, N);
      WRITELN;
      WRITE('Press the <ENTER> key. ');
      READLN(Ans);
      WRITELN;
      State := Computing;
      while (State = Computing) do
        begin
          WRITELN;
          WRITE('Want  to see  the  divided difference  table ? <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans = 'y') or (Ans = 'Y') then
            begin
              DDTABLE(D, X, N);
              WRITELN;
              WRITE('Press the <ENTER> key. ');
              READLN(Ans);
              WRITELN;
            end;
          EVALUATE(D, X, N, T, A0, B0, M);
          WRITELN;
          WRITE('Press the <ENTER> key. ');
          READLN(Ans);
          WRITELN;
          WRITELN;
          WRITE('Want to see  the  ordinary  polynomial  form ? <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans = 'y') or (Ans = 'Y') then
            begin
              MAKEXPOLY(D, X, C, N);
              PRINTXPOLY(C, N);
              WRITELN;
            end;
          WRITELN;
          WRITE('Do you want to evaluate the polynomial again ? <Y/N>  ');
          READLN(Resp);
          WRITELN;
          if (Resp <> 'Y') and (Resp <> 'y') then
            State := Done;
          if (Resp = 'Y') or (Resp = 'y') then
            CLRSCR;
        end;
      WRITELN;
      WRITELN;
      WRITE('Want to make a different  Newton  polynomial ? <Y/N>  ');
      READLN(Resp);
      WRITELN;
      if (Resp = 'Y') or (Resp = 'y') then
        Stat := More
      else
        Stat := Done;
    end;
end.                                               {End Main Program}

